#ifndef _CONSOLE_H
#define _CONSOLE_H

#include <stdarg.h>

#include "check.h"

/**
 * \addtogroup Console Serial Console
 *
 * \brief In- and output using a serial console
 *
 * This adds the ability to print and read from a connected serial console (via USB).
 * You can use the primitives supplied by this module or even printf and scanf.
 *
 * In Atmel Studio 7 select Menu "Tools" and choose "Data Visualizer".
 * Click on sidebar "Configuration"
 * In frame "Modules" expand "External Connection" and select "Serial Port"
 * Configure "Serial Port Control Panel":
 * - choose the correct interface (something like "mEDBG Virtual COM Port (COM4)"
 * - set the "Baud rate" to the same value as the "BAUD_RATE" macro (by default 38400)
 * - choose correct "Parity" (default "disabled") and "Stop bits" (default "1")
 * - check "DTR" and "Open Terminal" (and "RTS" on slow systems)
 * Click "Connect"
 *
 * In Linux you can try to connect to the console using the libspicboard make target "console":
 *
 * \warning Cannot be used in conjunction with any other serial communication (including \ref COM )
 *
 * \warning Using printf/scanf (and derived functions) will consume a noticeable amount of flash memory!
 *
 * @{
 * \file console.h
 * \version \$Rev: 16347 $
 */

/**
 * \brief Parity bit types
 */
typedef enum {
	PARITY_DISABLED, /**< parity bit disabled */
	PARITY_EVEN,     /**< use even parity bit */
	PARITY_ODD       /**< use odd parity bit */
} __attribute__ ((__packed__)) CONSOLE_PARITY;

/**
 * \brief Connect to serial console
 *
 * Specify the baud rate, parity type and number of stop bits and initiate a
 * connection using the serial port (USART)
 *
 * \param baud     baud rate for serial connection
 * \param parity   pairty type (even/odd) if enabled
 * \param stopbits number of stop bits (1 or 2)
 * \retval  0  connection successfully set up
 * \retval -1  Baud rate achieved is higher than allowed
 * \retval -2  Baud rate achieved is lower than allowed
 * \retval -3  Baud rate value overflow
 * \retval -4; invalid stop bit count
 */
int8_t sb_console_connect(uint32_t baud, CONSOLE_PARITY parity, uint8_t stopbits);

/**
 * \brief Connect to serial console with default settings
 *
 * Connect with 38400 baud, no parity and single stop bit to a serial console.
 * 
 * This is automatically used for the sb_console_* functions 
 * if sb_console_connect was not called before.
 *
 * \retval  0  connection successfully set up
 * \retval -1  Baud rate achieved is higher than allowed
 * \retval -2  Baud rate achieved is lower than allowed
 * \retval -3  Baud rate value overflow
 * \retval -4; invalid stop bit count
 */
int8_t sb_console_connect_default(void);

/**
 * \brief Read the next character
 *
 * Invalid read operations (end of file) will return '\0'
 *
 * \retval character read from console
 */
char sb_console_getChar();

/**
 * \brief Read a line into the buffer
 *
 * This function reads in at most one less than size characters and stores them into the buffer pointed to by the char array.
 * Reading stops after an EOF or a newline. If a newline is read, it is stored into the buffer.
 * A terminating null byte (\0) is stored after the last character in the buffer.
 *
 * \param string pointer to buffer
 * \param size size of buffer
 * \retval  string on success
 * \retval  NULL on error or when end of file occurs while no characters have been read
 */
char * sb_console_getString(char *string, uint16_t size);

/**
 * \brief Prints a single character
 *
 * \param character  character to print
 * \retval -1  on error
 * \retval  0  on success
 */
int8_t sb_console_putChar(char character);


/**
 * \brief Writes the string and a trailing newline.
 *
 * \param string  string to print
 * \retval -1  on error
 * \retval  0  on success
 */
int8_t sb_console_putString(const char *string);

/**
 * \brief Writes the string and a trailing newline.
 *
 * \param string  string from flash memory to print
 * \retval -1  on error
 * \retval  0  on success
 */
int8_t sb_console_putStringFromFlash(const __flash char *string);

/** @}*/

#endif

