#ifndef LIBEZS_WAVE_H
#define LIBEZS_WAVE_H

#include <stdint.h>
#include <stdio.h>

/*!
 * \brief EZS Wave error numbers
 */
typedef enum {
	WAVE_FILE_NOT_FOUND = -1,
	WAVE_ERROR = -2,
	WAVE_EOF = -3,
	WAVE_OK = 0,
} EZS_Wave_Error_t;

typedef struct EZS_Wave {
	/********************************************************************************/
	/*                                  public data                                 */
	/* filled in by ezs_wopen()                                                     */
	/********************************************************************************/
	/*!
	 * \brief compression code
	 */
	int16_t format_tag;
	/*!
	 * \brief number of channels
	 */
	int16_t channels;
	/*!
	 * \brief sampling rate
	 */
	uint32_t sample_rate;
	/*!
	 * \brief average number of bytes per second
	 */
	uint32_t avg_bytes_sec;
	/*!
	 * \brief number of samples per sample slice
	 */
	int16_t block_align;
	/*!
	 * \brief bits per sample
	 */
	int16_t bits_per_sample;
	int16_t bytes_per_sample;
	/*!
	 * \brief number of samples left in the wave stream
	 */
	uint32_t data_size;

	/********************************************************************************/
	/*                                  private data                                */
	/********************************************************************************/
	/*!
	 * \brief handle for the wave file. Not part of the public interface.
	 */
	FILE *file;
} EZS_Wave;

/*!
 * \brief open a wave file
 * \param wave pointer to wave context
 * \param filename filename of the wave file
 * \return error number, WAVE_OK otherwise
 */
EZS_Wave_Error_t ezs_wopen(EZS_Wave *wave, char filename[12]);

/*!
 * \brief read samples from a wave file.
 *
 * each call to ezs_wread writes wave->channels * wave->bytes_per_sample
 * bytes into buffer.
 *
 * \param wave pointer to a wave file context that has been initialized using ezs_wopen().
 * \return WAVE_EOF when end of file is reached, Error number on error, or WAVE_OK otherwise.
 * \param buffer pointer to buffer for read data. Must be big enough
 * to hold wave->channels samples.
 */
EZS_Wave_Error_t ezs_wread(EZS_Wave *wave, void *buffer);

/*!
 * \brief cleanup function. Call it when you are done using the wave context.
 * \param wave pointer to a wave context that has been initialized using ezs_wopen().
 * \return error number on error, WAVE_OK otherwise
 */
EZS_Wave_Error_t ezs_wclose(EZS_Wave *wave);

/*!
 * \brief prints information from the wave's header to stdout
 */
void ezs_wprint_header(EZS_Wave *wave);


#endif // LIBEZS_WAVE_H
