#include "ezs_sb16.h"

#if 0
#include <cyg/kernel/kapi.h>
#endif
#include <cyg/hal/hal_io.h>
#include <cyg/hal/hal_if.h>
#include <assert.h>
#include <stdio.h>

/*!
 * \brief write only, used to reset the DSP to its default state.
 */
const int16_t DSP_RESET_OFFSET   = 0x6;

/*!
 * \brief read only, use to access in-bound DSP data.
 */
const int16_t DSP_READ_OFFSET    = 0xA;

/*!
 * \brief write: send commands or data to the DSP.
 *        read: indicates whether the DSP is ready to accept commands or data.
 */
const int16_t DSP_WRITE_OFFSET   = 0xC;

/*!
 * \brief read only, indicates whether there is any in-bound data available for reading.
 */
const int16_t DSP_STATUS_OFFSET  = 0xE;

const int16_t DSP_INT_ACK_OFFSET = 0xF;
const int16_t MIXER_ADDR_OFFSET  = 0x4;
const int16_t MIXER_DATA_OFFSET  = 0x5;

/*
 * DSP commands
 */
const uint8_t DSP_PLAY_SAMPLE        = 0x10;
const uint8_t DSP_RECORD_SAMPLE      = 0x20;
const uint8_t DSP_8BIT_DMA_PAUSE     = 0xD0;
const uint8_t DSP_8BIT_DMA_CONTINUE  = 0xD4;

const uint8_t DSP_16BIT_DMA_PAUSE    = 0xD5;
const uint8_t DSP_16BIT_DMA_CONTINUE = 0xD6;

const uint8_t DSP_16BIT_DMA_EXIT     = 0xD9;
const uint8_t DSP_8BIT_DMA_EXIT      = 0xDA;

const uint8_t DSP_VERSION            = 0xE1;

const uint8_t DSP_READY              = 0xAA;
const uint8_t DSP_SPEAKER_ON         = 0xD1;
const uint8_t DSP_SPEAKER_OFF        = 0xD3;

const uint8_t DSP_SET_OUTPUT_RATE    = 0x41;
const uint8_t DSP_SET_INPUT_RATE     = 0x42;

const uint8_t DSP_SET_DMA_AUTO_8BIT_INPUT   = 0x2C;
const uint8_t DSP_SET_DMA_AUTO_8BIT_OUTPUT  = 0x1C;

const uint8_t DSP_SET_BLOCK_TRANSFER_SIZE = 0x48;

/*
 * mixer commands
 */
const uint8_t MIXER_MASTER_LEFT       = 0x30;
const uint8_t MIXER_MASTER_RIGHT      = 0x31;
const uint8_t MIXER_VOC_LEFT          = 0x32;
const uint8_t MIXER_VOC_RIGHT         = 0x33;
const uint8_t MIXER_MIDI_LEFT         = 0x34;
const uint8_t MIXER_MIDI_RIGHT        = 0x35;
const uint8_t MIXER_CD_LEFT           = 0x36;
const uint8_t MIXER_CD_RIGHT          = 0x37;
const uint8_t MIXER_LINE_LEFT         = 0x38;
const uint8_t MIXER_LINE_RIGHT        = 0x39;
const uint8_t MIXER_MIC_VOLUME        = 0x3A;
const uint8_t MIXER_PC_SPEAKER        = 0x3B;
const uint8_t MIXER_OUTPUT_SWITCHES   = 0x3C;
const uint8_t MIXER_INPUT_LEFT        = 0x3D;
const uint8_t MIXER_INPUT_RIGHT       = 0x3E;
const uint8_t MIXER_INPUT_GAIN_LEFT   = 0x3F;
const uint8_t MIXER_INPUT_GAIN_RIGHT  = 0x40;
const uint8_t MIXER_OUTPUT_GAIN_LEFT  = 0x41;
const uint8_t MIXER_OUTPUT_GAIN_RIGHT = 0x42;
const uint8_t MIXER_AGC               = 0x43;
const uint8_t MIXER_TREBLE_LEFT       = 0x44;
const uint8_t MIXER_TREBLE_RIGHT      = 0x45;
const uint8_t MIXER_BASS_LEFT         = 0x46;
const uint8_t MIXER_BASS_RIGHT        = 0x47;

void ezs_sb16_read_DSP_version(SB16 *sb);
void ezs_sb16_wait_for_DSP_readable(SB16 *sb);
void ezs_sb16_wait_for_DSP_writable(SB16 *sb);

void ezs_sb16_print(SB16 *sb)
{
  printf("EZS SB16: addr=0x%x intr=%d DMA=%d DMA16=%d\n",
      sb->io_addr, sb->interrupt, sb->DMA_no, sb->h_DMA_no);
  printf("EZS SB16: reset=0x%x read=0x%x write=0x%x status=0x%x mixer_addr=0x%x DSP_int_ack=0x%x\n",
      sb->DSP_reset, sb->DSP_read, sb->DSP_write, sb->DSP_status, sb->mixer_addr, sb->mixer_data, sb->DSP_int_ack);
  printf("EZS SB16: version %d.%d\n", sb->major_version, sb->minor_version);
}

void ezs_sb16_init(SB16 *sb,
                   int16_t io_addr,
                   int16_t interrupt,
                   int16_t DMA,
                   int16_t h_DMA)
{
  sb->io_addr     = io_addr;
  sb->interrupt   = interrupt;

  sb->DMA_no      = DMA;
  sb->h_DMA_no    = h_DMA;

  sb->DSP_reset   = sb->io_addr + DSP_RESET_OFFSET;
  sb->DSP_read    = sb->io_addr + DSP_READ_OFFSET;
  sb->DSP_write   = sb->io_addr + DSP_WRITE_OFFSET;
  sb->DSP_status  = sb->io_addr + DSP_STATUS_OFFSET;
  sb->mixer_addr  = sb->io_addr + MIXER_ADDR_OFFSET;
  sb->mixer_data  = sb->io_addr + MIXER_DATA_OFFSET;
  sb->DSP_int_ack = sb->io_addr + DSP_INT_ACK_OFFSET;

  ezs_sb16_reset_DSP(sb);
  ezs_sb16_write_DSP(sb, DSP_SPEAKER_ON);
  ezs_sb16_print(sb);
}

void ezs_sb16_read_DSP_version(SB16 *sb)
{
  ezs_sb16_wait_for_DSP_writable(sb);
  HAL_WRITE_UINT8(sb->DSP_write, DSP_VERSION);

  sb->major_version = ezs_sb16_read_DSP(sb);
  sb->minor_version = ezs_sb16_read_DSP(sb);

  assert(sb->major_version == 4 && "not a Sound Blaster 16");
}

void ezs_sb16_reset_DSP(SB16 *sb)
{
  HAL_WRITE_UINT8(sb->DSP_reset, 1);
#if 0
  /* TODO: sleep for 3 microseconds */
  CYGACC_CALL_IF_DELAY_US(1000);
#endif
  HAL_WRITE_UINT8(sb->DSP_reset, 0);

  ezs_sb16_wait_for_DSP_readable(sb);

  {
    int p = 0;
    for(p = 0; p < 1000; ++p) {
      uint8_t value = 0;
      HAL_READ_UINT8(sb->DSP_read, value);
      if (value == DSP_READY) {
        ezs_sb16_read_DSP_version(sb);
        return;
      }
    }
  }

  assert(false && "EZS SB16: Unable to Reset the DSP!");
}

void ezs_sb16_acknowledge_irq(SB16 *sb)
{
  uint8_t value = 0;

  HAL_READ_UINT8(sb->DSP_status, value);
}

int ezs_sb16_is_DSP_ready_for_read(SB16 *sb)
{
  uint8_t value = 0;

  HAL_READ_UINT8(sb->DSP_status, value);

  return (value & (1 << 7)) != 0;
}

int ezs_sb16_is_DSP_ready_for_write(SB16 *sb)
{
  uint8_t value = 0;

  HAL_READ_UINT8(sb->DSP_write, value);

  return (value & (1 << 7)) == 0;
}

void ezs_sb16_wait_for_DSP_readable(SB16 *sb)
{
  int not_ready = 1;
  while(not_ready) {
    int ready = ezs_sb16_is_DSP_ready_for_read(sb);
    not_ready = !ready;
  }
}

void ezs_sb16_wait_for_DSP_writable(SB16 *sb)
{
  while(!ezs_sb16_is_DSP_ready_for_write(sb));
}

void ezs_sb16_write_DSP(SB16 *sb, uint8_t value)
{
  ezs_sb16_wait_for_DSP_writable(sb);

  HAL_WRITE_UINT8(sb->DSP_write, value);
}

uint8_t ezs_sb16_read_DSP(SB16 *sb)
{
  ezs_sb16_wait_for_DSP_readable(sb);

  uint8_t value = 0;
  HAL_READ_UINT8(sb->DSP_read, value);

  return value;
}

void ezs_sb16_set_mixer_setting(SB16 *sb, uint8_t target, uint8_t value)
{
  HAL_WRITE_UINT8(sb->mixer_addr, target);
  HAL_WRITE_UINT8(sb->mixer_data, value);
}

void ezs_sb16_set_mixer_settings(SB16 *sb, SB16_Mixer *settings)
{
  uint8_t *pointer = &settings->master_left;

  uint8_t index = 0;
  for (index = MIXER_MASTER_LEFT; index <= MIXER_MIC_VOLUME; ++index) {
    ezs_sb16_set_mixer_setting(sb, index, (*pointer << 3));
  }

  ezs_sb16_set_mixer_setting(sb, MIXER_PC_SPEAKER, settings->pc_speaker << 6);

  pointer = &settings->output_switches;
  for (index = MIXER_OUTPUT_SWITCHES; index <= MIXER_INPUT_RIGHT; ++index) {
    ezs_sb16_set_mixer_setting(sb, index, *pointer);
  }

  pointer = &settings->input_gain_left;
  for (index = MIXER_INPUT_GAIN_LEFT; index <= MIXER_OUTPUT_GAIN_RIGHT; ++index) {
    ezs_sb16_set_mixer_setting(sb, index, *pointer);
  }

  ezs_sb16_set_mixer_setting(sb, MIXER_AGC, settings->agc);

  pointer = &settings->treble_left;
  for (index = MIXER_TREBLE_LEFT; index <= MIXER_BASS_RIGHT; ++index) {
    ezs_sb16_set_mixer_setting(sb, index, *pointer);
  }
}

uint8_t ezs_sb16_get_mixer_setting(SB16 *sb, uint8_t source)
{
  HAL_WRITE_UINT8(sb->mixer_addr, source);

  uint8_t value = 0;
  HAL_READ_UINT8(sb->mixer_data, value);

  if (source >= 0x30 && source <= 0x3A) return value >> 3;
  if (source >= 0x3F && source <= 0x42) return value >> 6;
  if (source >= 0x44 && source <= 0x47) return value >> 4;
}

void ezs_sb16_get_mixer_settings(SB16 *sb, SB16_Mixer *settings)
{
  uint8_t *pointer = &settings->master_left;

  {
    uint8_t index = 0;
    for (index = 0x30; index <= 0x47; ++index) {
      *pointer = ezs_sb16_get_mixer_setting(sb, index);
      ++pointer;
    }
  }
}

void ezs_sb16_play_sample(SB16 *sb, uint8_t sample)
{
  ezs_sb16_write_DSP(sb, DSP_PLAY_SAMPLE);
  ezs_sb16_write_DSP(sb, sample);
}

void ezs_sb16_play_dac1(SB16 *sb, uint8_t sample)
{
  ezs_sb16_write_DSP(sb, 0x11);
  ezs_sb16_write_DSP(sb, sample);
}

uint8_t ezs_sb16_record_sample(SB16 *sb)
{
  ezs_sb16_write_DSP(sb, DSP_RECORD_SAMPLE);
  return ezs_sb16_read_DSP(sb);
}


#if 0
void get_buffers(void)
{
  /* 8-bit */
  buf_8 = mem_8 = malloc(2*bufferlength);
  if (mem_8 == NULL)
    exit(EXIT_FAILURE);

  if (((getlinearaddr(mem_8) % 65536) + bufferlength) > 65536)
    buf_8 += 1;

  blockptr_8[0] = buf_8;
  blockptr_8[1] = buf_8 + blocklength;

  buffer8_addr = getlinearaddr(buf_8);
  buffer8_page = buffer8_addr / 65536;
  buffer8_ofs  = buffer8_addr % 65536;

  _fmemset(buf_8, 0x80, bufferlength);

  /* 16-bit */
  buf_16 = mem_16 = malloc(4*bufferlength);
  if (mem_16 == NULL)
    exit(EXIT_FAILURE);

  if ((((getlinearaddr(mem_16) >> 1) % 65536) + bufferlength) > 65536)
    buf_16 += 1;

  blockptr_16[0] = buf_16;
  blockptr_16[1] = buf_16 + blocklength;

  buffer16_addr = getlinearaddr(buf_16);
  buffer16_page = buffer16_addr / 65536;
  buffer16_ofs  = (buffer16_addr >> 1) % 65536;

  _fmemset(buf_16, 0x00, 2*bufferlength);

  curblock_8  = 0;
  curblock_16 = 0;

  intcount_8  = 0;
  intcount_16 = 0;
  intcount    = 0;
}

#endif


void ezs_sb16_set_output_rate(SB16 *sb, uint16_t rate)
{
  assert(rate >= 5000 && "sampling rate must be at least 5000 Hz");
  assert(rate <= 44100 && "sampling rate must be at most 44100 Hz");

  ezs_sb16_write_DSP(sb, DSP_SET_OUTPUT_RATE);
  ezs_sb16_write_DSP(sb, ezs_high_byte(rate));
  ezs_sb16_write_DSP(sb, ezs_low_byte(rate));
}

void ezs_sb16_setup_mode(
        SB16 *sb,
        void *buffer,
        uint16_t block_size)
{
#if 0
  /* Sound Blaster Pro 2 way of doing this */
    ezs_sb16_write_DSP(sb, DSP_SET_BLOCK_TRANSFER_SIZE);

    uint16_t dsp_block_size = block_size - 1;
    ezs_sb16_write_DSP(sb, ezs_low_byte(dsp_block_size));
    ezs_sb16_write_DSP(sb, ezs_high_byte(dsp_block_size));

    ezs_sb16_write_DSP(sb, DSP_SET_DMA_AUTO_8BIT_OUTPUT);
#endif

    ezs_sb16_write_DSP(sb, 0xC6); /* 8 bit DMA auto init without FIFO */
    ezs_sb16_write_DSP(sb, 0x30); /* stereo unsigned */
    uint16_t dsp_block_size = block_size - 1;
    ezs_sb16_write_DSP(sb, ezs_low_byte(dsp_block_size));
    ezs_sb16_write_DSP(sb, ezs_high_byte(dsp_block_size));
}

void ezs_sb16_setup_DMA(
    SB16 *sb,
    void *buffer,
    uint16_t block_size)
{
  ezs_dma_init(&sb->DMA);
#if 0
  cyg_interrupt_mask(sb->interrupt + 32);
#endif
  ezs_dma_set_mode_byte(&sb->DMA,
      sb->DMA_no,
      EZS_DMA_Demand,
      EZS_DMA_Increment,
      EZS_DMA_Auto_Initialization,
      EZS_DMA_Read);
  ezs_dma_disable_channel(&sb->DMA);
  ezs_dma_clear_flip_flop(&sb->DMA);
  ezs_dma_write_mode_byte(&sb->DMA);

  uint8_t page = (((uint32_t) buffer) >> 16) & 0xFF;
  ezs_dma_set_buffer_info(&sb->DMA, page, buffer, block_size);
#if 0
  cyg_interrupt_unmask(sb->interrupt + 32);
#endif
  ezs_dma_enable_channel(&sb->DMA);
}

void ezs_sb16_sound_on(SB16 *sb)
{
  HAL_WRITE_UINT8(sb->DSP_write, 0xD6);
}

void ezs_sb16_sound_off(SB16 *sb)
{
  HAL_WRITE_UINT8(sb->DSP_write, 0xD5);
}

void ezs_sb16_exit_8bit_DMA(SB16 *sb)
{
  ezs_sb16_write_DSP(sb, DSP_8BIT_DMA_PAUSE);
  ezs_sb16_write_DSP(sb, DSP_8BIT_DMA_EXIT);
  ezs_dma_disable_channel(&sb->h_DMA);
}

void ezs_sb16_exit_16bit_DMA(SB16 *sb)
{
  ezs_sb16_write_DSP(sb, DSP_16BIT_DMA_PAUSE);
  ezs_sb16_write_DSP(sb, DSP_16BIT_DMA_EXIT);
  ezs_dma_disable_channel(&sb->DMA);
}

void ezs_sb16_stop_sound(SB16 *sb)
{
  ezs_sb16_exit_8bit_DMA(sb);
  ezs_sb16_exit_16bit_DMA(sb);
}
